% Copyright (c) 2013, Massachusetts Institute of Technology
% This program was presented in the book "Visual Psychophysics:
% From Laboratory to Theory" by Zhong-Lin Lu and Barbara Dosher.
% The book is available at http://mitpress.mit.edu/books/visual-psychophysics

%%% Program: RemoveFirstOrderContamination.m

function RemoveFirstOrderContamination

%% Display Setup Module

%% Display Setup Module

% Define display parameters

whichScreen = max(Screen('screens'));
p.ScreenDistance = 30;  % in inches
p.ScreenHeight = 15;    % in inches
p.ScreenGamma = 2;      % from monitor calibration
p.maxLuminance = 100; % from monitor calibration
p.ScreenBackground = 0.5;

% Open the display window and hide the mouse cursor

if exist('onCleanup', 'class'), oC_Obj = onCleanup(@()sca); end % close any pre-existing PTB Screen window
PsychImaging('PrepareConfiguration');
PsychImaging('AddTask', 'General', 'FloatingPoint32Bit');   % set up a 32-bit framebuffer
PsychImaging('AddTask', 'General', 'NormalizedHighresColorRange');
PsychImaging('AddTask', 'FinalFormatting', 'DisplayColorCorrection', 'SimpleGamma');  % setup Gamma correction method
[windowPtr p.ScreenRect] = PsychImaging('OpenWindow', whichScreen, p.ScreenBackground);  % open a display window
PsychColorCorrection('SetEncodingGamma', windowPtr, 1 / p.ScreenGamma);  % set Gamma for all color channels
HideCursor;

% Get frame rate and set screen font

p.ScreenFrameRate = FrameRate(windowPtr);
Screen('TextFont', windowPtr, 'Times');
Screen('TextSize', windowPtr, 24);

%% Experimental Module

% Specify the stimulus
p.alpha = 0.7 : 0.1 : 1.1;  % set up alpha values
p.radius = 6;               % image radius in visual degree
p.innerRad = 1; 
p.contrast1 = 0.1;          % 1st order signal contrast
p.contrast2 = 0.5;          % 2nd order signal contrast
p.noise1 = 0.5;             % 1st order noise contrast
p.noise2 = 1;               % 2nd order noise contrast
p.nFactor = 2;              % noise pixel size
p.sf = 6;                   % cycles per degree
p.fixSize = 0.4;            % fixation size in deg
p.fixColor = 0;             % fixation color
p.tf = 7.5;                 % rotations per second
p.ITI = 0.5;                % seconds between trials
repeats = 16;               % Number of trials to repeat 
                            % for each alpha
keys = {'left' 'right' 'esc'}; % keys to respond for 
                               % direction, and to break

% Compute stimulus parameters
ppd = pi/180 * p.ScreenDistance / p.ScreenHeight * ...
      p.ScreenRect(4); % pixels per degree
m = round(p.radius * 2 * ppd / p.nFactor) * p.nFactor; 
                            % stimulus size in pixels
fixRect=CenterRect([0 0 1 1] * p.fixSize * ppd, ...
      p.ScreenRect);
nAlpha = numel(p.alpha);    % number of alpha levels
nTrials = repeats * nAlpha; % total number of trials
nFrames = round(p.ScreenFrameRate / p.tf / 4); 
                            % # of refreshes for each image
dt = (nFrames - 0.5) / p.ScreenFrameRate; 
                            % time for Flip wait
p.randSeed = ClockRandSeed; % use clock to set random 
                            % number generator
[x, y] = meshgrid(linspace(-1, 1, m) * p.radius); 
                            % coodinates
[theta, r] = cart2pol(x, y);% transform x,y coordinates to
                            % polar coordinates
clip = r < p.innerRad | r > p.radius; % set up annulus
nPixels = m / p.nFactor;
bNoise = ones(nPixels) * 0.5;
bNoise(1 : numel(bNoise) / 2) = -0.5; 
                            % binary noise -0.5 or 0.5
tex = zeros (1, 5);

% Initialize a table to set up experimental conditions
p.recLabel = {'trialIndex' 'alphaIndex' 'clockwise' ...
      'respCorrect' 'respTime'};
rec = nan(nTrials, length(p.recLabel)); 
        % matrix rec is nTrials x 5 of NaN
rec(:, 1) = 1 : nTrials;    % count  trial numbers from 1 
                            % to nTrials
alphaIndex = repmat(1 : nAlpha, repeats, 1);
clockwise = ones(repeats, nAlpha); 
                            % first set all to 1: clockwise
clockwise(1 : repeats / 2, :) = -1; 
                 % change first half to -1: counter-cw
[rec(:, 2) ind] = Shuffle(alphaIndex(:)); 
                            % shuffle alpha index
rec(:, 3) = clockwise(ind); % shuffle clockwise in the same
                            % order

% Prioritize display to optimize display timing
Priority(MaxPriority(windowPtr));

% Start experiment with instructions
str = ['Press left or right arrow keys for\n\n' ...
       'counter-clockwiset or clockwise rotation.\n\n' ...
       'Press SPACE to start.'];
DrawFormattedText(windowPtr, str, 'center', 'center', 1);
% Draw Instruction text string centered in window
Screen('Flip', windowPtr);  
        % flip the text image into active buffer
WaitTill('space');         % wait till space bar is pressed
Secs = Screen('Flip', windowPtr);
p.start = datestr(now);    % record start time

% Run nTrials trials
for i = 1 : nTrials
    % Make textures for each frame
    for j = 1 : 5 
        bNoise(:) = Shuffle(bNoise(:)); 
              % new noise image in each frame
        nImg = Expand(bNoise, p.nFactor);
        ang = (j - 1) * pi / 2 / p.sf * rec(i, 3); 
              % phase shift: 90 deg per frame
        wedge = sin(p.sf * (theta + ang)) / 2; % [-0.5 0.5]
        if mod(j, 2) % odd number frames: 2nd order
            img = nImg * p.noise2  ...
                  .* (wedge * p.contrast2 + 0.5);
        else  % even number frames: 1st order
            img = nImg * p.noise1 ...
                  + wedge * p.contrast1;
        end
        ind = img < 0; % find pixels with negative contrast
        img(ind) = img(ind) * p.alpha(rec(i, 2)); 
                     % apply alpha
        img(clip) = 0;
        tex(j) = Screen('MakeTexture', windowPtr, img + ...
                 p.ScreenBackground, 0, 0, 2);
    end
    
    [key vbl] = WaitTill(Secs + p.ITI); 
    
    % Display each frame
    for j = 1 : 5
        Screen('DrawTexture', windowPtr, tex(j));
        Screen('FillOval', windowPtr, p.fixColor, fixRect);
        vbl = Screen('Flip', windowPtr, vbl + dt); 
    end
    Screen('FillOval', windowPtr, p.fixColor, fixRect);
    Screen('Flip', windowPtr, vbl + dt); 
           % turn off last frame
    Screen('Close', tex); % close textures of each trial
    [key Secs] = WaitTill(keys);   % wait till response
    if iscellstr(key), key = key{1}; end 
           % take the first in case of multiple keys
    if strcmp(key, 'esc'), break; end % to stop
    rec(i, 4) = strcmp(key, 'right') == (rec(i, 3) == 1); 
           % record correctness
    rec(i, 5) = Secs - vbl; % record respTime
end
p.finish = datestr(now); % record finish time
save RemoveFirstOrderContamination_rst rec p;  
        % save the results
 
%% System Reinstatement Module
Priority(0);  % restore priority
sca;          % close window and textures, 
              % restore color lookup table
% Plot the result
pCorrect = zeros(nAlpha, 1);
for i = 1 : nAlpha
    foo = rec(rec(: ,2) == i, 4);
    foo(isnan(foo)) = [];
    pCorrect(i) = sum(foo) / numel(foo);
end
figure(3);
plot(p.alpha, pCorrect * 100, '+-');
axis([0.5 1.1 0 108]);
xlabel('Alpha');
ylabel('Percent Correct')


